// Copyright (c) 2001-2005 Extended Systems, Inc.
// Portions Copyright (c) 2005-2006, iAnywhere Solutions, Inc.
// All rights reserved. All unpublished rights reserved.
//
// This source code can be used, modified, or copied by the licensee as long as
// the modifications (or the new binary resulting from a copy or modification of
// this source code) are used with Extended Systems' products. The source code
// is not redistributable as source code, but is redistributable as compiled
// and linked binary code. If the source code is used, modified, or copied by
// the licensee, Extended Systems Inc. reserves the right to receive from the
// licensee, upon request, at no cost to Extended Systems Inc., the modifications.
//
// Extended Systems Inc. does not warrant that the operation of this software
// will meet your requirements or that the operation of the software will be
// uninterrupted, be error free, or that defects in software will be corrected.
// This software is provided "AS IS" without warranty of any kind. The entire
// risk as to the quality and performance of this software is with the purchaser.
// If this software proves defective or inadequate, purchaser assumes the entire
// cost of servicing or repair. No oral or written information or advice given
// by an Extended Systems Inc. representative shall create a warranty or in any
// way increase the scope of this warranty.

/*******************************************************************************
* Source File  : adsbackup.c
* Description  : Command line backup utility for use with Advantage Database
*                Server.  This utility packages up all the required information
*                to perform a backup or restore and executes the backup or
*                restore on the specified server.
* Notes        : This utility loads ACE32.DLL or libace.so dynamically so it can
*                be used with different versions of ACE.  The minimum ACE version
*                requirement is v6.0.  However, if the server OS is NetWare, the
*                ACE version must be 7.0 or greater.
*
*                To compile under linux use: gcc -ldl adsbackup.c
*******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <sys/timeb.h>

#if defined( WIN32 )

   #include <windows.h>
   #include <conio.h>

   #define ACE32_MODULE_NAME  "ace32.dll"

#elif defined( __linux__ )

   #include <dlfcn.h>
   #include <errno.h>
   #include <termios.h>
   #include <sys/time.h>
   #include <string.h>
   #include <stdlib.h>

   #ifndef FALSE
      #define FALSE           0
   #endif

   #ifndef TRUE
      #define TRUE            1
   #endif

   #define ACE

   typedef void*              HMODULE;
   #define ACE32_MODULE_NAME  "libace.so"
   #define GetLastError()     (errno)
   #define stricmp( x, y )    strcasecmp( x, y )
#else
   #error WIN32 or __linux__ not defined
#endif

#include "ace.h"

/* ACE API function pointers */
ADSCONNECT60_PTR                 gpfnAdsConnect60;
ADSDISCONNECT_PTR                gpfnAdsDisconnect;
ADSGETLASTERROR_PTR              gpfnAdsGetLastError;
ADSCREATESQLSTATEMENT_PTR        gpfnAdsCreateSQLStatement;
ADSSTMTSETTABLETYPE_PTR          gpfnAdsStmtSetTableType;
ADSSTMTSETTABLECHARTYPE_PTR      gpfnAdsStmtSetTableCharType;
ADSSTMTSETTABLELOCKTYPE_PTR      gpfnAdsStmtSetTableLockType;
ADSSTMTSETTABLERIGHTS_PTR        gpfnAdsStmtSetTableRights;
ADSEXECUTESQLDIRECT_PTR          gpfnAdsExecuteSQLDirect;
ADSREGISTERCALLBACKFUNCTION_PTR  gpfnAdsRegisterCallbackFunction;
ADSGETLONG_PTR                   gpfnAdsGetLong;

/* Option variables */
UNSIGNED8  gucDatabaseMode = FALSE;        // TRUE if we're operating on a database, FALSE if free tables
UNSIGNED8  gucDontOverwrite = FALSE;       // TRUE if we don't want to overwrite files already backed up
UNSIGNED8  gucMetadataOnly = FALSE;        // TRUE if we only want to backup the DD metadata
UNSIGNED8  gucDiff = FALSE;                // TRUE if we're doing a differential backup
UNSIGNED8  gucPrepareDiff = FALSE;         // TRUE if we're preparing a differential backup
UNSIGNED8  gucRestore = FALSE;             // TRUE if we're doing a restore, FALSE for doing a backup
UNSIGNED8  gucVerbose = FALSE;             // TRUE to turn on Verbose mode for more output information
UNSIGNED8  gucDebugSnapshot = FALSE;       // TRUE to perform a debug snapshot
SIGNED32   glSeverity = 1;                 // Minimum severity level of errors to report (default to 1 to get all)
UNSIGNED8  *gpucInitialTime = NULL;
UNSIGNED8  *gpucReleaseTime = NULL;
UNSIGNED8  *gpucFinalTime = NULL;
UNSIGNED8  *gpucSourcePath = NULL;         // Source path of backup or restore
UNSIGNED8  *gpucDestinationPath = NULL;    // Destination path of backup or restore
UNSIGNED8  *gpucIncludeList = NULL;        // List of tables in include in a backup
UNSIGNED8  *gpucExcludeList = NULL;        // List of tables to exclude from a backup
UNSIGNED8  *gpucFreeTablePasswords = NULL; // Free table password(s)
UNSIGNED8  *gpucPassword = NULL;           // DD user password in database mode, free table password in free table mode
UNSIGNED8  *gpucOptions = NULL;            // Options string for Backup/Restore SQL stmt
UNSIGNED8  *gpucFileMask = NULL;           // File mask for free table backups, ie *.adt
UNSIGNED8  *gpucOutputFile = NULL;         // Path+filename of output file
UNSIGNED8  *gpucOutputPath = NULL;         // Path to store output file (filename determined by adsbackup)
UNSIGNED8  *gpucConnectionPath = NULL;     // Connection path of ADS server to perform backup or restore on
UNSIGNED16 gusTableType = ADS_ADT;         // Table type of SQL statement
UNSIGNED16 gusCharType = ADS_ANSI;         // Character type of SQL statement
UNSIGNED16 gusLockType = ADS_PROPRIETARY_LOCKING; // Locking type of SQL statement
UNSIGNED16 gusRightsChecking = ADS_CHECKRIGHTS; // Rights checking mode of SQL statement


/* Version information string for ADSVER.EXE utility */
char *gpcIdAxsRevisionStr = "EsIAx!@# 8.10";


#if defined( __linux__ )
/*******************************************************************************
*  Module       : LoadLibrary
*  Created      : 11/26/2001
*  Last Mod     :
*  Return       : handle of the shared object
*  Desc         : load a shared object
*  Notes        : win compatibility layer
*******************************************************************************/
void *LoadLibrary( char *pucLibName )
{
   return dlopen( pucLibName, RTLD_NOW );
}



/*******************************************************************************
*  Module       : FreeLibrary
*  Created      : 11/26/2001
*  Last Mod     :
*  Return       : non-zero on success, 0 on error
*  Desc         : unload a shared object
*  Notes        : win compatibility layer
*******************************************************************************/
unsigned short FreeLibrary( void *hLib )
{
   /* return boolean value */
   return !(dlclose( hLib ));
}



/*******************************************************************************
*  Module       : GetProcAddress
*  Created      : 11/26/2001
*  Last Mod     :
*  Return       : pointer to function
*  Desc         : get pointer to function pucFunc
*  Notes        : win compatibility layer
*******************************************************************************/
void *GetProcAddress( void *hLib, char *pucFunc )
{
   return dlsym( hLib, pucFunc );
}



static struct termios stored_settings;
/*******************************************************************************
* Module      : set_keypress
* Created     : 6/24/2005
* Last Mod    :
* Return      :
* Description : Configures the terminal to return control after reading one byte.
*               Also set it to not echo the input to look nicer.
* Notes       : Derived from the comp.unix.programmer FAQ
********************************************************************************/
void set_keypress(void)
{
    struct termios new_settings;

    tcgetattr(0,&stored_settings);

    new_settings = stored_settings;

    /* Disable canonical mode */
    new_settings.c_lflag &= (~ICANON);
    /* Also disable echo as we don't want to see the input */
    new_settings.c_lflag &= (~ECHO);

    new_settings.c_cc[VTIME] = 0;
    new_settings.c_cc[VMIN] = 1;

    tcsetattr(0,TCSANOW,&new_settings);
    return;
} /* set_keypress */



/*******************************************************************************
* Module      : reset_keypress
* Created     : 6/24/2005
* Last Mod    :
* Return      :
* Description : Reset the terminal to it's original settings
* Notes       : Derived from the comp.unix.programmer FAQ
********************************************************************************/
void reset_keypress(void)
{
    tcsetattr(0,TCSANOW,&stored_settings);
    return;
} /* reset_keypress */

#endif /* __linux__ */



/*******************************************************************************
* Module      : HitEscape
* Created     : 6/22/2005
* Last Mod    :
* Return      : TRUE or FALSE
* Description : Returns TRUE if the ESCAPE key has been hit
* Notes       :
********************************************************************************/
UNSIGNED8 HitEscape( void )
{
   int iHit = 0;

#ifdef WIN32
   iHit = kbhit();
   if ( iHit )
      iHit = getch();
#else
   fd_set rfds;
   struct timeval tv;

   FD_ZERO( &rfds );
   FD_SET( 0, &rfds );

   /* Wait for zero seconds */
   tv.tv_sec = 0;
   tv.tv_usec = 0;

   /* Check stdin for any input */
   if ( select( 1, &rfds, NULL, NULL, &tv ))
      iHit = getchar();
#endif

   return iHit == 27 ? TRUE : FALSE;
} /* HitEscape */



/*******************************************************************************
* Module      : ShowPercentage
* Created     : 6/22/2005
* Last Mod    :
* Return      : 0 or 1
* Description : Progress callback function for use with AdsExecuteSQLDirect
* Notes       : Return zero to continue the SQL query, return non-zero to abort it.
********************************************************************************/
UNSIGNED32 WINAPI ShowPercentage( UNSIGNED16 usPercentDone, UNSIGNED32 ulCallbackID )
{
   // If the ESCAPE key has been struck, return a non-zero value to abort the operation
   if ( HitEscape() )
      return 1;

   printf( "\rPercent Complete: %2d%%  Press ESC to abort.", usPercentDone );
   return 0;

}  /* ShowPercentage */



/*******************************************************************************
* Module      : PrintUsage
* Created     : 6/21/2005
* Last Mod    :
* Return      :
* Description : Print the list of accepted options and command line arguments
* Notes       :
********************************************************************************/
void PrintUsage( void )
{
   printf( "Advantage Database Server Backup Utility\n" );
   printf( "Usage:\n" );
   printf( "       Free Table Backup\n" );
   printf( "          adsbackup [options] <src path> <file mask> <dest path>\n" );
   printf( "       Data Dictionary Backup\n" );
   printf( "          adsbackup [options] <src database> <dest path>\n" );
   printf( "Valid options include:\n" );
   printf( "   -a  Prepare a database for a differential backup\n" );
   printf( "   -c[ANSI|OEM]  Character type (default ANSI)\n" );
   printf( "   -d  Don't overwrite existing backup tables\n" );
   printf( "   -e<file1, .. ,filen>  Exclude file list\n" );
   printf( "   -f  Differential backup\n" );
   printf( "   -h[ON|OFF]  Rights checking (default ON)\n" );
   printf( "   -i<file1, .. ,filen>  Include file list\n" );
   printf( "   -m  Backup metadata only\n" );
   printf( "   -n<path>  Path to store the backup output table\n" );
   printf( "   -o<filepath>  Path and filename to the backup output table\n" );
   printf( "   -p  Password(s): Database user password if source path is a database\n" );
   printf( "                    List of free table passwords if source path is a directory\n" );
   printf( "                    Free table usage can pass a single password for all\n" );
   printf( "                    encrypted tables, or a name/value pair for each table, for\n" );
   printf( "                    example, \"table1=pass1;table2=pass2\"\n" );
   printf( "   -q[PROP|COMPAT]  Locking mode, proprietary or compatible (default PROP)\n" );
   printf( "   -r  Restore database or free table files\n" );
   printf( "   -s<server path>  Connection path of ADS server to perform backup or restore\n" );
// printf( "   -t<server port>  Connection port of ADS server to perform backup or restore\n" ); // Option 't' used by JDBC backup utility only
   printf( "   -u[ADT|CDX|NTX]  Table type (default ADT)\n" );
   printf( "   -v[1-10]  Lowest level of error severity to return (default 1)\n" );
   printf( "\n" );
   printf( "Examples:\n" );
#ifdef __linux__
   printf( "   adsbackup -pPass \\\\\\\\server\\\\share\\\\mydata.add \\\\\\\\server\\\\share\\\\backup\n" );
   printf( "   adsbackup -pTablePass \\\\\\\\server\\\\share *.adt \\\\\\\\server\\\\share\\\\backup\n" );
   printf( "   adsbackup -r -pPass \\\\\\\\server\\\\share\\\\backup\\\\mydata.add \\\\\\\\server\\\\share\\\\backup\\\\mydata.add\n" );
   printf( "   adsbackup -r -pTablePass \\\\\\\\server\\\\share\\\\backup \\\\\\\\server\\\\share\n" );
#else
   printf( "   adsbackup -pPass \\\\server\\share\\mydata.add \\\\server\\share\\backup\n" );
   printf( "   adsbackup -pTablePass \\\\server\\share *.adt \\\\server\\share\\backup\n" );
   printf( "   adsbackup -r -pPass \\\\server\\share\\backup\\mydata.add \\\\server\\share\\backup\\mydata.add\n" );
   printf( "   adsbackup -r -pTablePass \\\\server\\share\\backup \\\\server\\share\n" );
#endif
   exit( 0 );
} /* PrintUsage */



/*******************************************************************************
* Module      : GetEntryPoint
* Created     : 6/21/2005
* Last Mod    :
* Return      : zero on success, otherwise last error number
* Description : Get a specific ACE entrypoint via GetProcAddress
* Notes       :
********************************************************************************/
UNSIGNED32 GetEntryPoint( HMODULE hAce32, char *pcEntrypointName, void **ppfnEntrypoint )
{
   *ppfnEntrypoint = GetProcAddress( hAce32, pcEntrypointName );
   if ( *ppfnEntrypoint == NULL )
      return GetLastError();
   else
      return AE_SUCCESS;
} /* GetEntryPoint */



/*******************************************************************************
* Module      : GetAdsEntrypoints
* Created     : 6/21/2005
* Last Mod    :
* Return      : zero on success, otherwise last error number
* Description : Get all required ACE entrypoint function pointers
* Notes       :
********************************************************************************/
UNSIGNED32 GetAdsEntrypoints( HMODULE hAce32 )
{
   UNSIGNED32 ulRetCode;

   ulRetCode = GetEntryPoint( hAce32, "AdsConnect60", (void**)&gpfnAdsConnect60 );
   if ( ulRetCode || ( gpfnAdsConnect60 == NULL ))
      return ulRetCode;

   ulRetCode = GetEntryPoint( hAce32, "AdsDisconnect", (void**)&gpfnAdsDisconnect );
   if ( ulRetCode || ( gpfnAdsDisconnect == NULL ))
      return ulRetCode;

   ulRetCode = GetEntryPoint( hAce32, "AdsCreateSQLStatement", (void**)&gpfnAdsCreateSQLStatement );
   if ( ulRetCode || ( gpfnAdsCreateSQLStatement == NULL ))
      return ulRetCode;

   ulRetCode = GetEntryPoint( hAce32, "AdsStmtSetTableType", (void**)&gpfnAdsStmtSetTableType );
   if ( ulRetCode || ( gpfnAdsStmtSetTableType == NULL ))
      return ulRetCode;

   ulRetCode = GetEntryPoint( hAce32, "AdsStmtSetTableLockType", (void**)&gpfnAdsStmtSetTableLockType );
   if ( ulRetCode || ( gpfnAdsStmtSetTableLockType == NULL ))
      return ulRetCode;

   ulRetCode = GetEntryPoint( hAce32, "AdsStmtSetTableCharType", (void**)&gpfnAdsStmtSetTableCharType );
   if ( ulRetCode || ( gpfnAdsStmtSetTableCharType == NULL ))
      return ulRetCode;

   ulRetCode = GetEntryPoint( hAce32, "AdsStmtSetTableRights", (void**)&gpfnAdsStmtSetTableRights );
   if ( ulRetCode || ( gpfnAdsStmtSetTableRights == NULL ))
      return ulRetCode;

   ulRetCode = GetEntryPoint( hAce32, "AdsExecuteSQLDirect", (void**)&gpfnAdsExecuteSQLDirect );
   if ( ulRetCode || ( gpfnAdsExecuteSQLDirect == NULL ))
      return ulRetCode;

   ulRetCode = GetEntryPoint( hAce32, "AdsGetLastError", (void**)&gpfnAdsGetLastError );
   if ( ulRetCode || ( gpfnAdsGetLastError == NULL ))
      return ulRetCode;

   ulRetCode = GetEntryPoint( hAce32, "AdsRegisterCallbackFunction", (void**)&gpfnAdsRegisterCallbackFunction );
   if ( ulRetCode || ( gpfnAdsRegisterCallbackFunction == NULL ))
      return ulRetCode;

   ulRetCode = GetEntryPoint( hAce32, "AdsGetLong", (void**)&gpfnAdsGetLong );
   if ( ulRetCode || ( gpfnAdsGetLong == NULL ))
      return ulRetCode;

   return AE_SUCCESS;
} /* GetAdsEntrypoints */



/*******************************************************************************
* Module      : ParseOptions
* Created     : 6/21/2005
* Last Mod    :
* Return      :
* Description : Parse the command line options and arguments into the global
*               option & argument variables.
* Notes       :
********************************************************************************/
UNSIGNED32 ParseOptions( SIGNED8 argc, SIGNED8 **argv )
{
   SIGNED8    cCounter;
   UNSIGNED32 ulAllocationSize;

   for ( cCounter = 1; cCounter < argc; cCounter++ )
      {
      // All options should be preceded by a '-'
      if ( argv[cCounter][0] == '-' )
         {
         switch ( argv[cCounter][1] )
            {
            case 'a':
            case 'A':
               gucPrepareDiff = TRUE;
               break;

            case 'b':
            case 'B':
               gucDebugSnapshot = TRUE;
               break;

            case 'c':
            case 'C':
               if ( !stricmp( argv[cCounter] + 2, "ANSI" ))
                  gusCharType = ADS_ANSI;
               else if ( !stricmp( argv[cCounter] + 2, "OEM" ))
                  gusCharType = ADS_OEM;
               else
                  {
                  printf( "Invalid character type: %s\n", argv[cCounter] + 2 );
                  return AE_INVALID_OPTION;
                  }
               break;

            case 'd':
            case 'D':
               gucDontOverwrite = TRUE;
               break;

            case 'e':
            case 'E':
               gpucExcludeList = &argv[cCounter][2];
               break;

            case 'f':
            case 'F':
               gucDiff = TRUE;
               break;

            case 'g':
            case 'G':
               gucVerbose = TRUE;
               break;

            case 'h':
            case 'H':
               if ( !stricmp( argv[cCounter] + 2, "ON" ))
                  gusRightsChecking = ADS_CHECKRIGHTS;
               else if ( !stricmp( argv[cCounter] + 2, "OFF" ))
                  gusRightsChecking = ADS_IGNORERIGHTS;
               else
                  {
                  printf( "Invalid rights checking option: %s\n", argv[cCounter] + 2 );
                  return AE_INVALID_OPTION;
                  }
               break;

            case 'i':
            case 'I':
               gpucIncludeList = &argv[cCounter][2];
               break;

            case 'j':
            case 'J':
               gpucReleaseTime = &argv[cCounter][2];
               break;

            case 'k':
            case 'K':
               gpucFinalTime = &argv[cCounter][2];
               break;

            case 'l':
            case 'L':
               gpucInitialTime = &argv[cCounter][2];
               break;

            case 'm':
            case 'M':
               gucMetadataOnly = TRUE;
               break;

            case 'n':
            case 'N':
               gpucOutputPath = &argv[cCounter][2];
               break;

            case 'o':
            case 'O':
               gpucOutputFile = &argv[cCounter][2];
               break;

            case 'p':
            case 'P':
               gpucPassword = &argv[cCounter][2];
               break;

            case 'q':
            case 'Q':
               if ( !stricmp( argv[cCounter] + 2, "PROP" ))
                  gusLockType = ADS_PROPRIETARY_LOCKING;
               else if ( !stricmp( argv[cCounter] + 2, "COMPAT" ))
                  gusLockType = ADS_COMPATIBLE_LOCKING;
               else
                  {
                  printf( "Invalid locking type: %s\n", argv[cCounter] + 2 );
                  return AE_INVALID_OPTION;
                  }
               break;

            case 'r':
            case 'R':
               gucRestore = TRUE;

               /*
                * If the source and destination paths have already been
                * parsed, the destination path may be incorrectly stored.
                */
               if ( gpucFileMask )
                  {
                  gpucDestinationPath = gpucFileMask;
                  gpucFileMask = NULL;
                  }
               break;

            case 's':
            case 'S':
               gpucConnectionPath = &argv[cCounter][2];
               break;

/*
            case 't':
            case 'T':
                Option 't' used by the JDBC client only
               break;
*/

            case 'u':
            case 'U':
               if ( !stricmp( argv[cCounter] + 2, "ADT" ))
                  gusTableType = ADS_ADT;
               else if ( !stricmp( argv[cCounter] + 2, "CDX" ))
                  gusTableType = ADS_CDX;
               else if ( !stricmp( argv[cCounter] + 2, "NTX" ))
                  gusTableType = ADS_NTX;
               else
                  {
                  printf( "Invalid table type: %s\n", argv[cCounter] + 2 );
                  return AE_INVALID_OPTION;
                  }
               break;

            case 'v':
            case 'V':
               glSeverity = atol( &argv[cCounter][2] );
               break;

            case '?':
               PrintUsage();
               break;

            default:
               printf( "Unrecognized option: %s\n", argv[cCounter] );
               return AE_INVALID_OPTION;
            }
         }
      else
         {
         /* Not an option, must be the source or destination path or file mask */
         if ( gpucSourcePath == NULL )
            gpucSourcePath = argv[cCounter];
         else
            {
            /* Do a quick check to see if we're really doing a DB or free table operation */
            if (( !gucDatabaseMode ) && ( stricmp( (char*)&gpucSourcePath[ strlen( gpucSourcePath ) - 4 ], ".ADD" ) == 0 ))
                  gucDatabaseMode = TRUE;

             if (( gucDatabaseMode == FALSE ) &&
                  ( gucRestore == FALSE ) &&
                  ( gpucFileMask == NULL ))
               gpucFileMask = argv[cCounter];
             else if ( gpucDestinationPath )
                {
                printf( "Invalid option, too many paths given: %s\n", argv[cCounter] );
                return AE_INVALID_OPTION;
                }
             else
                gpucDestinationPath = argv[cCounter];
            }
         }
      }

   ulAllocationSize = ( gpucIncludeList ? ( strlen( gpucIncludeList ) + 9 ) : 0 ) +
                      ( gpucExcludeList ? ( strlen( gpucExcludeList ) + 9 ) : 0 ) +
                      ( gucDontOverwrite ? 14 : 0 ) +
                      ( gucMetadataOnly ? 9 : 0 ) +
                      ( gucPrepareDiff ? 12 : 0 ) +
                      ( gucDiff ? 5 : 0 ) +
                      ( gucDebugSnapshot ? 14 : 0 ) +
                      ( gucVerbose ? 8 : 0 ) +
                      ( gpucInitialTime ? ( strlen( gpucInitialTime ) + 13 ) : 0 ) +
                      ( gpucReleaseTime ? ( strlen( gpucReleaseTime ) + 13 ) : 0 ) +
                      ( gpucFinalTime ? ( strlen( gpucFinalTime ) + 11 ) : 0 );

   if ( ulAllocationSize )
      {
      // Allocate 1 extra for the NULL terminator
      gpucOptions = malloc( ulAllocationSize + 1 );
      if ( gpucOptions == NULL )
         {
         printf( "Memory allocation failure.\n" );
         return AE_ALLOCATION_FAILED;
         }

      // Start with an empty string
      gpucOptions[0] = 0x00;

      if ( gpucIncludeList )
         {
         strcat( gpucOptions, "INCLUDE=" );
         strcat( gpucOptions, gpucIncludeList );
         strcat( gpucOptions, ";" );
         }

      if ( gpucExcludeList )
         {
         strcat( gpucOptions, "EXCLUDE=" );
         strcat( gpucOptions, gpucExcludeList );
         strcat( gpucOptions, ";" );
         }

      if ( gucDontOverwrite )
         strcat( gpucOptions, "DONTOVERWRITE;" );

      if ( gucMetadataOnly )
         strcat( gpucOptions, "METAONLY;" );

      if ( gucPrepareDiff )
         strcat( gpucOptions, "PREPAREDIFF;" );

      if ( gucDiff )
         strcat( gpucOptions, "DIFF;" );

      if ( gucDebugSnapshot )
         strcat( gpucOptions, "DEBUGSNAPSHOT;" );

      if ( gucVerbose )
         strcat( gpucOptions, "VERBOSE;" );

      if ( gpucInitialTime )
         {
         strcat( gpucOptions, "INITIALTIME=" );
         strcat( gpucOptions, gpucInitialTime );
         strcat( gpucOptions, ";" );
         }

      if ( gpucReleaseTime )
         {
         strcat( gpucOptions, "RELEASETIME=" );
         strcat( gpucOptions, gpucReleaseTime );
         strcat( gpucOptions, ";" );
         }

      if ( gpucFinalTime )
         {
         strcat( gpucOptions, "FINALTIME=" );
         strcat( gpucOptions, gpucFinalTime );
         strcat( gpucOptions, ";" );
         }

      }

#ifdef _DEBUG
   /* Debug check to make sure option string is the correct size */
   if ( ulAllocationSize && gpucOptions &&
        ( ulAllocationSize != strlen( gpucOptions )))
      printf( "Option string size is wrong.\n" );
#endif

   return AE_SUCCESS;
} /* ParseOptions */



/*******************************************************************************
* Module      : GetOutputTableName
* Created     : 6/23/2005
* Last Mod    :
* Return      :
* Description : Generate the backup output file name
* Notes       : Generates a filename based on the current date & time in this
*               format: CCYYMMDDHHMMSS.  Uses provided path if specified by the
*               -n option.
********************************************************************************/
UNSIGNED8 *GetOutputTableName( UNSIGNED8 *pucOutputTable )
{
   struct timeb stTimeb;
   struct tm   *pstTmPtr;

   ftime( &stTimeb );
   pstTmPtr = localtime( ( time_t * ) &(stTimeb.time ));
   if ( pstTmPtr == NULL )
      {
      printf( "An Error occurred while getting the current time: %d\n", GetLastError() );
      return NULL;
      }

   if ( gpucOutputPath )
      {
      if (( gpucOutputPath[ strlen( gpucOutputPath ) ] == '\\' ) ||
          ( gpucOutputPath[ strlen( gpucOutputPath ) ] == '/' ))
         sprintf( (char*)pucOutputTable, "%s%s_%.4d%.2d%.2d%.2d%.2d%.2d.%s",
                  gpucOutputPath,
                  gucRestore ? "restore" : "backup",
                  1900L + pstTmPtr->tm_year,
                  pstTmPtr->tm_mon + 1,
                  pstTmPtr->tm_mday,
                  pstTmPtr->tm_hour,
                  pstTmPtr->tm_min,
                  pstTmPtr->tm_sec,
                  gusTableType == ADS_ADT ? "adt" : "dbf" );
      else
         sprintf( (char*)pucOutputTable, "%s%c%s_%.4d%.2d%.2d%.2d%.2d%.2d.%s",
                  gpucOutputPath,
#ifdef __linux__
                  '/',
#else
                  '\\',
#endif
                  gucRestore ? "restore" : "backup",
                  1900L + pstTmPtr->tm_year,
                  pstTmPtr->tm_mon + 1,
                  pstTmPtr->tm_mday,
                  pstTmPtr->tm_hour,
                  pstTmPtr->tm_min,
                  pstTmPtr->tm_sec,
                  gusTableType == ADS_ADT ? "adt" : "dbf" );

      }
   else
      sprintf( (char*)pucOutputTable, "%s_%.4d%.2d%.2d%.2d%.2d%.2d.%s",
               gucRestore ? "restore" : "backup",
               1900L + pstTmPtr->tm_year,
               pstTmPtr->tm_mon + 1,
               pstTmPtr->tm_mday,
               pstTmPtr->tm_hour,
               pstTmPtr->tm_min,
               pstTmPtr->tm_sec,
               gusTableType == ADS_ADT ? "adt" : "dbf" );

   return pucOutputTable;
} /* GetOutputTableName */



/*******************************************************************************
* Module      : CheckForBackupError
* Created     : 6/23/2005
* Last Mod    :
* Return      :
* Description : Search the backup output table for errors worth returning from
*               main()
* Notes       : Backup severity values are:
*                                            HIGH        10
*                                            MEDHIGH     7
*                                            MED         5
*                                            LOW         1
*                                            NONE        0
********************************************************************************/
UNSIGNED32 CheckForBackupError( ADSHANDLE hResults )
{
   UNSIGNED32 ulRetCode;
   UNSIGNED32 ulBackupError;

   // Attempt to get the first severe error code
   ulRetCode = (*gpfnAdsGetLong)( hResults, "ErrorCode", (SIGNED32*)&ulBackupError );

   // If no records pass the filter, then return no error
   if ( ulRetCode == AE_NO_CURRENT_RECORD )
      return AE_SUCCESS;

   // If AdsGetLong failed for some other reason, return that error
   if ( ulRetCode )
      return ulRetCode;

   // Return the severe error we found in the top record
   return ulBackupError;
} /* CheckForBackupError */



/*******************************************************************************
* Module      : CreateSQLText
* Created     : 6/23/2005
* Last Mod    :
* Global Vars :
* Return      : AE_ALLOCATION_FAILED or AE_SUCCESS
* Description : Allocate memory and create the SQL statment text for executing
*               the backup or restore query.
* Notes       : This function determines the allocation size based on the length
*               of the current SQL script text.  If you add more text to the script
*               you must also increase the allocation size.
********************************************************************************/
UNSIGNED32 CreateSQLText( UNSIGNED8 **ppucSQLText )
{
   UNSIGNED32 ulAllocationSize;

   // Calculate how long the SQL statement will be
   if ( gucRestore )
      {
      // Restore
      if ( gucDatabaseMode )
         // Database Restore
         ulAllocationSize = 524 + strlen( gpucSourcePath ) +
                            ( gpucPassword ? strlen( gpucPassword ) : 0 ) +
                            strlen( gpucDestinationPath ) +
                            ( gpucOptions ? strlen( gpucOptions ) : 0 ) +
                            ( strlen( gpucOutputFile ) * 3 );
      else
         // Free Table Restore
         ulAllocationSize = 524 + strlen( gpucSourcePath ) +
                            strlen( gpucDestinationPath ) +
                            ( gpucOptions ? strlen( gpucOptions ) : 0 ) +
                            ( gpucPassword ? strlen( gpucPassword ) : 0 ) +
                            ( strlen( gpucOutputFile ) * 3 );
      }
   else
      {
      // Backup
      if ( gucDatabaseMode )
         // Database Backup
         ulAllocationSize = 514 + strlen( gpucDestinationPath ) +
                            ( gpucOptions ? strlen( gpucOptions ) : 0 ) +
                            ( strlen( gpucOutputFile ) * 3 );
      else
         // Free Table Backup
         ulAllocationSize = 526 + strlen( gpucSourcePath ) +
                            strlen( gpucFileMask ) +
                            strlen( gpucDestinationPath ) +
                            ( gpucOptions ? strlen( gpucOptions ) : 0 ) +
                            ( gpucPassword ? strlen( gpucPassword ) : 0 ) +
                            ( strlen( gpucOutputFile ) * 3 );
      }

   // Allocate enough memory to contain the necessary SQL text
   *ppucSQLText = malloc( ulAllocationSize + 1 );
   if ( *ppucSQLText == NULL )
      {
      printf( "Memory allocation failure.\n" );
      return AE_ALLOCATION_FAILED;
      }

   // First create the specific SQL text
   if ( gucRestore )
      {
      // Restore
      if ( gucDatabaseMode )
         // Database Restore
         sprintf( (char*)*ppucSQLText,
                  "DECLARE cur Cursor;" \
                  "OPEN cur AS EXECUTE PROCEDURE sp_RestoreDatabase( '%s', '%s', '%s', '%s' );",
                  gpucSourcePath,
                  gpucPassword ? gpucPassword : (UNSIGNED8*)"",
                  gpucDestinationPath,
                  gpucOptions ? gpucOptions : (UNSIGNED8*)"" );
      else
         // Free Table Restore
         sprintf( (char*)*ppucSQLText,
                  "DECLARE cur Cursor;" \
                  "OPEN cur AS EXECUTE PROCEDURE sp_RestoreFreeTables( '%s', '%s', '%s', '%s' );",
                  gpucSourcePath,
                  gpucDestinationPath,
                  gpucOptions ? gpucOptions : (UNSIGNED8*)"",
                  gpucPassword ? gpucPassword : (UNSIGNED8*)"" );
      }
   else
      {
      // Backup
      if ( gucDatabaseMode )
         // Database Backup
         sprintf( (char*)*ppucSQLText,
                  "DECLARE cur Cursor;" \
                  "OPEN cur AS EXECUTE PROCEDURE sp_BackupDatabase( '%s', '%s' );",
                  gpucDestinationPath,
                  gpucOptions ? gpucOptions : (UNSIGNED8*)"" );
      else
         // Free Table Backup
         sprintf( (char*)*ppucSQLText,
                  "DECLARE cur Cursor;" \
                  "OPEN cur AS EXECUTE PROCEDURE sp_BackupFreeTables( '%s', '%s', '%s', '%s', '%s' );",
                  gpucSourcePath,
                  gpucFileMask,
                  gpucDestinationPath,
                  gpucOptions ? gpucOptions : (UNSIGNED8*)"",
                  gpucPassword ? gpucPassword : (UNSIGNED8*)"" );
      }

   // Finally, complete the SQL script
   // NOTE: If you add more text to any of this SQL script be sure to increase the allocation size above.
   sprintf( &(*ppucSQLText)[strlen( *ppucSQLText)],
            "CREATE TABLE [%s] ( [Severity] Integer, [ErrorCode] Integer, [ErrorMsg] Memo,"\
            "[TableName] Memo, [MoreInfo] Memo, [SourceFile] char(32), " \
            "[SourceLine] Integer ) AS FREE TABLE;" \
      "WHILE FETCH cur DO " \
      "INSERT INTO [%s] VALUES( cur.[Severity], cur.[Error Code], cur.[Error Message], "\
      "cur.[Table Name], cur.[Additional Info], cur.[Source File], cur.[Source Line] );" \
      "END;" \
      "CLOSE cur;" \
      "SELECT [ErrorCode] FROM [%s] WHERE [Severity] >= %d",
      gpucOutputFile,
      gpucOutputFile,
      gpucOutputFile,
      glSeverity );

   return AE_SUCCESS;
} /* CreateSQLText */



#define CHECK_ADS_ERROR( ulErr )                                  \
   if ( ulErr != AE_SUCCESS )                                     \
      {                                                           \
      usErrorLen = sizeof( aucError );                            \
      (*gpfnAdsGetLastError)( &ulRetCode, aucError, &usErrorLen );\
      printf( "%s\n", aucError );                                 \
      goto mainExit;                                              \
      }

/*******************************************************************************
* Module      : main
* Created     : 6/23/2005
* Last Mod    :
* Return      : Errors >= minimum severity if found (default severity = 1, LOW)
* Description : Main function of the AdsBackup utility
* Notes       :
********************************************************************************/
int main( SIGNED8 argc, SIGNED8 **argv )
{
   HMODULE    hAce32 = 0;
   UNSIGNED32 ulRetCode;
   ADSHANDLE  hConnect = 0;
   ADSHANDLE  hResults;
   ADSHANDLE  hStmt;
   UNSIGNED8  aucError[ ADS_MAX_ERROR_LEN + 1 ];
   UNSIGNED8  *pucSQLText = NULL;
   UNSIGNED16 usErrorLen;
   UNSIGNED8  aucOutputTable[ ADS_MAX_PATH + 1 ];

   // A minimum of two arguments are required to do anything (not including the exe name)
   if ( argc < 3 )
      PrintUsage();

#ifdef __linux__
   /* For Linux, configure the terminal for use with the callback function */
   set_keypress();
#endif

   ulRetCode = ParseOptions( argc, argv );
   if ( ulRetCode )
      goto mainExit;

   // Load ACE32.DLL dynamically
   hAce32 = LoadLibrary( ACE32_MODULE_NAME );
   if ( hAce32 == NULL )
      {
      ulRetCode = GetLastError();
      printf( "Failed to load %s.  Last error: %d\n", ACE32_MODULE_NAME, ulRetCode );
      goto mainExit;
      }

   /* Get the required ACE API function pointers */
   ulRetCode = GetAdsEntrypoints( hAce32 );
   if ( ulRetCode != AE_SUCCESS )
      {
      printf( "Failed get all required entrypoints.  Last error: %d\n", ulRetCode );
      goto mainExit;
      }

   // Get a connection to the Advantage server
   if ( gpucConnectionPath )
      ulRetCode = (*gpfnAdsConnect60)( gpucConnectionPath, ADS_REMOTE_SERVER, gucDatabaseMode ? "ADSSYS" : "",
                                       gucDatabaseMode ? gpucPassword : NULL, ADS_DEFAULT,
                                       &hConnect );
   else
      ulRetCode = (*gpfnAdsConnect60)( gpucSourcePath, ADS_REMOTE_SERVER, gucDatabaseMode ? "ADSSYS" : "",
                                       gucDatabaseMode ? gpucPassword : NULL, ADS_DEFAULT,
                                       &hConnect );

   CHECK_ADS_ERROR( ulRetCode );

   // Create an SQL statement to work with
   ulRetCode = (*gpfnAdsCreateSQLStatement)( hConnect, &hStmt );
   CHECK_ADS_ERROR( ulRetCode );

   /* Set the table type if it isn't the default */
   if ( gusTableType != ADS_ADT )
      {
      ulRetCode = (*gpfnAdsStmtSetTableType)( hStmt, gusTableType );
      CHECK_ADS_ERROR( ulRetCode );
      }

   /* Set the character type if it isn't the default */
   if ( gusCharType != ADS_ANSI )
      {
      ulRetCode = (*gpfnAdsStmtSetTableCharType)( hStmt, gusCharType );
      CHECK_ADS_ERROR( ulRetCode );
      }

   /* Set the lock type if it isn't the default */
   if ( gusLockType != ADS_PROPRIETARY_LOCKING )
      {
      ulRetCode = (*gpfnAdsStmtSetTableLockType)( hStmt, gusLockType );
      CHECK_ADS_ERROR( ulRetCode );
      }

   /* Set the rights checking flag if it isn't the default */
   if ( gusRightsChecking != ADS_CHECKRIGHTS )
      {
      ulRetCode = (*gpfnAdsStmtSetTableRights)( hStmt, gusRightsChecking );
      CHECK_ADS_ERROR( ulRetCode );
      }

   // Register a callback function to provide percentage information and abort ability
   ulRetCode = (*gpfnAdsRegisterCallbackFunction)( ShowPercentage, 1 );
   CHECK_ADS_ERROR( ulRetCode );

   // If no output file was specified, create a new output file path
   if ( gpucOutputFile == NULL )
      gpucOutputFile = GetOutputTableName( aucOutputTable );

   // If the output file is still NULL, an error has occurred
   if ( gpucOutputFile == NULL )
      goto mainExit;

   /* Allocate memory and create the SQL statement to perform the backup or restore */
   ulRetCode = CreateSQLText( &pucSQLText );
   if ( ulRetCode )
      goto mainExit;

   printf( "\n" );

   // Begin the backup or restore
   ulRetCode = (*gpfnAdsExecuteSQLDirect)( hStmt, pucSQLText, &hResults );
   CHECK_ADS_ERROR( ulRetCode );

   if ( gucRestore )
      printf( "\rPercent Complete: 100%%                     \nRestore Complete\n\n" );
   else
      printf( "\rPercent Complete: 100%%                     \nBackup Complete\n\n" );

   // Look for a severe enough error in the output table
   if ( ulRetCode == AE_SUCCESS )
      ulRetCode = CheckForBackupError( hResults );

mainExit:
   if ( hConnect )
      (*gpfnAdsDisconnect)( hConnect );

   if ( hAce32 )
      FreeLibrary( hAce32 );

   if ( pucSQLText )
      free( pucSQLText );

   if ( gpucOptions )
      free( gpucOptions );

#ifdef __linux__
   /* For Linux, restore the original terminal configuration */
   reset_keypress();
#endif

   return ulRetCode;

} /* main */